package com.ejie.ab04b.service.dokusi;

import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.ejie.ab04b.constantes.ConstantesDokusi;
import com.ejie.ab04b.exception.AB04BParseException;
import com.ejie.ab04b.exception.DokusiException;
import com.ejie.ab04b.model.DocumentoAperturas;
import com.ejie.ab04b.model.DocumentoOS2;
import com.ejie.ab04b.model.DocumentoOS3;
import com.ejie.ab04b.util.DokusiUtils;
import com.ejie.ab04b.util.PropertiesUtil;
import com.ejie.ab04b.util.seguridad.UtilSeguridad;
import com.ejie.ab04b.webservice.client.dokusi.T65BFSDWSClasePort;
import com.ejie.t65.jaxb.beans.fsd.T65BAttributeListType;
import com.ejie.t65.jaxb.beans.fsd.T65BConditionListType;
import com.ejie.t65.jaxb.beans.fsd.T65BContentType;
import com.ejie.t65.jaxb.beans.fsd.T65BDocumentIDListType;
import com.ejie.t65.jaxb.beans.fsd.T65BDocumentIDType;
import com.ejie.t65.jaxb.beans.fsd.T65BDocumentType;
import com.ejie.t65.jaxb.beans.fsd.T65BFormatType;
import com.ejie.t65.jaxb.beans.fsd.T65BStringListType;

/**
 * 
 * @author GFI-NORTE
 * 
 */
@Service()
public class DokusiService {

	@Autowired()
	private T65BFSDWSClasePort t65bFSDWSClasePortQSPort;

	@Autowired()
	private Properties appConfiguration;

	private static final Logger LOGGER = LoggerFactory
			.getLogger(DokusiService.class);

	/**
	 * Almacena un documento en Dokusi
	 * 
	 *  auditUser
	 *            String
	 *  doc
	 *            DocumentoAperturas
	 * 
	 *  T65BDocumentIDType.
	 *
	 * @param auditUser the audit user
	 * @param doc the doc
	 * @return the t 65 B document ID type
	 * @throws DokusiException             excepción lanzada por Dokusi
	 * @throws AB04BParseException             excepción al parsear la respuesta de Dokusi
	 */
	public T65BDocumentIDType almacenarDocumento(String auditUser,
			DocumentoAperturas doc) throws DokusiException, AB04BParseException {

		DokusiService.LOGGER.info("almacenarDocumento");

		DokusiService.LOGGER.info("auditUser: {}", auditUser);

		T65BContentType t65bContentType = DokusiUtils.getInstance()
				.construirT65BContentType(doc);
		T65BAttributeListType t65bAttributeListType = DokusiUtils.getInstance()
				.generarMetadatosBasicos(doc);

		String content = "";
		if (t65bContentType != null) {
			content = DokusiUtils.getInstance().getT65BContentTypeAsString(
					t65bContentType);
		}
		DokusiService.LOGGER.info("content: {}", content);

		String attributeList = "";
		if (t65bAttributeListType != null) {
			attributeList = DokusiUtils.getInstance()
					.getT65BAttributeListTypeAsString(t65bAttributeListType);
		}
		DokusiService.LOGGER.info("attributeList: {}", attributeList);

		String response = null;
		try {
			response = this.t65bFSDWSClasePortQSPort.storeDocument(
					UtilSeguridad.getInstance().getTokenXLNets(), auditUser,
					content, doc.getTipoDocumento().getTipoDokusi091(),
					attributeList);
		} catch (Exception e) {
			DokusiService.LOGGER.error(
					"Error en la llamada a storeDocument de Dokusi", e);
			throw new DokusiException(e,
					" LOG NEGOCIO: Error en la llamada a storeDocument");
		}

		T65BDocumentIDType t65bDocumentIDType = DokusiUtils.getInstance()
				.getDokusiResponseAsT65BDocumentIDType(response);

		DokusiService.LOGGER.info(
				"Documento almacenado en Dokusi con OID {} y versión {}",
				t65bDocumentIDType.getId(), t65bDocumentIDType.getVersion());

		return t65bDocumentIDType;
	}

	/**
	 * Almacena un documento en Dokusi
	 * 
	 *  auditUser
	 *            String
	 *  doc
	 *            DocumentoAperturas
	 * 
	 *  T65BDocumentIDType.
	 *
	 * @param auditUser the audit user
	 * @param doc the doc
	 * @return the t 65 B document ID type
	 * @throws DokusiException             excepción lanzada por Dokusi
	 * @throws AB04BParseException             excepción al parsear la respuesta de Dokusi
	 */
	public T65BDocumentIDType almacenarDocumento(String auditUser,
			DocumentoOS2 doc) throws DokusiException, AB04BParseException {

		DokusiService.LOGGER.info("almacenarDocumento");

		DokusiService.LOGGER.info("auditUser: {}", auditUser);

		T65BContentType t65bContentType = DokusiUtils.getInstance()
				.construirT65BContentType(doc);
		T65BAttributeListType t65bAttributeListType = DokusiUtils.getInstance()
				.generarMetadatosBasicos(doc);

		String content = "";
		if (t65bContentType != null) {
			content = DokusiUtils.getInstance().getT65BContentTypeAsString(
					t65bContentType);
		}
		DokusiService.LOGGER.info("content: {}", content);

		String attributeList = "";
		if (t65bAttributeListType != null) {
			attributeList = DokusiUtils.getInstance()
					.getT65BAttributeListTypeAsString(t65bAttributeListType);
		}
		DokusiService.LOGGER.info("attributeList: {}", attributeList);

		String response = null;
		try {
			response = this.t65bFSDWSClasePortQSPort.storeDocument(
					UtilSeguridad.getInstance().getTokenXLNets(), auditUser,
					content, doc.getTipoDocumento().getTipoDokusi091(),
					attributeList);
		} catch (Exception e) {
			DokusiService.LOGGER.error(
					"Error en la llamada a storeDocument de Dokusi", e);
			throw new DokusiException(e,
					" LOG NEGOCIO: Error en la llamada a storeDocument");
		}

		T65BDocumentIDType t65bDocumentIDType = DokusiUtils.getInstance()
				.getDokusiResponseAsT65BDocumentIDType(response);

		DokusiService.LOGGER.info(
				"Documento almacenado en Dokusi con OID {} y versión {}",
				t65bDocumentIDType.getId(), t65bDocumentIDType.getVersion());

		return t65bDocumentIDType;
	}

	/**
	 * Almacena un documento en Dokusi
	 * 
	 *  auditUser
	 *            String
	 *  doc
	 *            DocumentoAperturas
	 * 
	 *  T65BDocumentIDType.
	 *
	 * @param auditUser the audit user
	 * @param doc the doc
	 * @return the t 65 B document ID type
	 * @throws DokusiException             excepción lanzada por Dokusi
	 * @throws AB04BParseException             excepción al parsear la respuesta de Dokusi
	 */
	public T65BDocumentIDType almacenarDocumento(String auditUser,
			DocumentoOS3 doc) throws DokusiException, AB04BParseException {

		DokusiService.LOGGER.info("almacenarDocumento");

		DokusiService.LOGGER.info("auditUser: {}", auditUser);

		T65BContentType t65bContentType = DokusiUtils.getInstance()
				.construirT65BContentType(doc);
		T65BAttributeListType t65bAttributeListType = DokusiUtils.getInstance()
				.generarMetadatosBasicos(doc);

		String content = "";
		if (t65bContentType != null) {
			content = DokusiUtils.getInstance().getT65BContentTypeAsString(
					t65bContentType);
		}
		DokusiService.LOGGER.info("content: {}", content);

		String attributeList = "";
		if (t65bAttributeListType != null) {
			attributeList = DokusiUtils.getInstance()
					.getT65BAttributeListTypeAsString(t65bAttributeListType);
		}
		DokusiService.LOGGER.info("attributeList: {}", attributeList);

		String response = null;
		try {
			response = this.t65bFSDWSClasePortQSPort.storeDocument(
					UtilSeguridad.getInstance().getTokenXLNets(), auditUser,
					content, doc.getTipoDocumento().getTipoDokusi091(),
					attributeList);
		} catch (Exception e) {
			DokusiService.LOGGER.error(
					"Error en la llamada a storeDocument de Dokusi", e);
			throw new DokusiException(e,
					" LOG NEGOCIO: Error en la llamada a storeDocument");
		}

		T65BDocumentIDType t65bDocumentIDType = DokusiUtils.getInstance()
				.getDokusiResponseAsT65BDocumentIDType(response);

		DokusiService.LOGGER.info(
				"Documento almacenado en Dokusi con OID {} y versión {}",
				t65bDocumentIDType.getId(), t65bDocumentIDType.getVersion());

		return t65bDocumentIDType;
	}

	/**
	 * Recupera un documento de Dokusi
	 * 
	 *  auditUser
	 *            String
	 *  doc
	 *            DocumentoAperturas
	 *  t65bStringListType
	 *            T65BStringListType
	 * 
	 *  T65BDocumentType.
	 *
	 * @param auditUser the audit user
	 * @param doc the doc
	 * @param t65bStringListType the t 65 b string list type
	 * @return the t 65 B document type
	 * @throws DokusiException             excepción lanzada por Dokusi
	 * @throws AB04BParseException             excepción al parsear la respuesta de Dokusi
	 */
	public T65BDocumentType recuperarDocumento(String auditUser,
			DocumentoAperturas doc, T65BStringListType t65bStringListType)
			throws DokusiException, AB04BParseException {

		DokusiService.LOGGER.info("recuperarDocumento");

		DokusiService.LOGGER.info("auditUser: {}", auditUser);

		DokusiService.LOGGER.info("doc: {}", doc.toString());

		T65BDocumentIDType t65bDocumentIDType = DokusiUtils.getInstance()
				.construirT65BDocumentIDType(doc.getOidDokusi065(), null);
		T65BContentType t65bContentType = new T65BContentType();

		String documentID = DokusiUtils.getInstance()
				.getT65BDocumentIDTypeAsString(t65bDocumentIDType);
		String keysList = DokusiUtils.getInstance()
				.getT65BStringListTypeAsString(t65bStringListType);

		String response = null;
		try {
			DokusiService.LOGGER
					.info("Se va a recuperar de Dokusi el formato del documento con OID {} y versión {}",
							t65bDocumentIDType.getId(),
							t65bDocumentIDType.getVersion());

			response = this.t65bFSDWSClasePortQSPort
					.retrieveDocumentFormats(UtilSeguridad.getInstance()
							.getTokenXLNets(doc.isMigrado()), auditUser,
							documentID);

			T65BFormatType format = DokusiUtils.getInstance()
					.getDokusiResponseAsT65BFormatListType(response)
					.getFormat().get(0);
			t65bContentType.setFormat(format);
			t65bContentType.setPifId(doc.getRutaPif065());

			String content = DokusiUtils.getInstance()
					.getT65BContentTypeAsString(t65bContentType);

			DokusiService.LOGGER
					.info("Se va a recuperar de Dokusi el documento con OID {} y versión {}",
							t65bDocumentIDType.getId(),
							t65bDocumentIDType.getVersion());

			response = this.t65bFSDWSClasePortQSPort
					.retrieveDocument(UtilSeguridad.getInstance()
							.getTokenXLNets(doc.isMigrado()), auditUser,
							documentID, keysList, content);
		} catch (Exception e) {
			DokusiService.LOGGER.error(
					"Error en la llamada a retrieveDocument de Dokusi", e);
			throw new DokusiException(e,
					" LOG NEGOCIO: Error en la llamada a retrieveDocument de Dokusi");
		}

		if (response != null) {
			T65BDocumentType t65bDocumentType = DokusiUtils.getInstance()
					.getDokusiResponseAsT65BDocumentType(response);

			DokusiService.LOGGER.info(
					"Documento recuperado de Dokusi con OID {} versión {}",
					t65bDocumentType.getDocumentID().getId(), t65bDocumentType
							.getDocumentID().getVersion());

			return t65bDocumentType;
		} else {
			DokusiService.LOGGER
					.warn("La respuesta de retrieveDocument de Dokusi es null");
			throw new DokusiException(
					" LOG NEGOCIO: La respuesta de retrieveDocument de Dokusi es null");
		}
	}

	/**
	 * Recupera un documento de Dokusi
	 * 
	 *  auditUser
	 *            String
	 *  doc
	 *            DocumentoAperturas
	 *  t65bStringListType
	 *            T65BStringListType
	 * 
	 *  T65BDocumentType.
	 *
	 * @param auditUser the audit user
	 * @param oid the oid
	 * @param t65bStringListType the t 65 b string list type
	 * @return the t 65 B document type
	 * @throws DokusiException             excepción lanzada por Dokusi
	 * @throws AB04BParseException             excepción al parsear la respuesta de Dokusi
	 */
	public T65BDocumentType recuperarDocumentoConOid(String auditUser,
			String oid, T65BStringListType t65bStringListType)
			throws DokusiException, AB04BParseException {

		DokusiService.LOGGER.info("recuperarDocumento");

		DokusiService.LOGGER.info("auditUser: {}", auditUser);

		DokusiService.LOGGER.info("oid: {}", oid);

		T65BDocumentIDType t65bDocumentIDType = DokusiUtils.getInstance()
				.construirT65BDocumentIDType(oid, null);
		T65BContentType t65bContentType = new T65BContentType();

		String documentID = DokusiUtils.getInstance()
				.getT65BDocumentIDTypeAsString(t65bDocumentIDType);
		String keysList = DokusiUtils.getInstance()
				.getT65BStringListTypeAsString(t65bStringListType);

		String response = null;
		try {
			DokusiService.LOGGER
					.info("Se va a recuperar de Dokusi el formato del documento con OID {} y versión {}",
							t65bDocumentIDType.getId(),
							t65bDocumentIDType.getVersion());

			response = this.t65bFSDWSClasePortQSPort.retrieveDocumentFormats(
					UtilSeguridad.getInstance().getTokenXLNets(false),
					auditUser, documentID);

			T65BFormatType format = DokusiUtils.getInstance()
					.getDokusiResponseAsT65BFormatListType(response)
					.getFormat().get(0);
			t65bContentType.setFormat(format);
			t65bContentType.setPifId(this.appConfiguration
					.getProperty(PropertiesUtil.PIF_RUTA_APP));

			String content = DokusiUtils.getInstance()
					.getT65BContentTypeAsString(t65bContentType);

			DokusiService.LOGGER
					.info("Se va a recuperar de Dokusi el documento con OID {} y versión {}",
							t65bDocumentIDType.getId(),
							t65bDocumentIDType.getVersion());

			response = this.t65bFSDWSClasePortQSPort.retrieveDocument(
					UtilSeguridad.getInstance().getTokenXLNets(false),
					auditUser, documentID, keysList, content);
		} catch (Exception e) {
			DokusiService.LOGGER.error(
					"Error en la llamada a retrieveDocument de Dokusi", e);
			throw new DokusiException(e,
					" LOG NEGOCIO: Error en la llamada a retrieveDocument de Dokusi");
		}

		if (response != null) {
			T65BDocumentType t65bDocumentType = DokusiUtils.getInstance()
					.getDokusiResponseAsT65BDocumentType(response);

			DokusiService.LOGGER.info(
					"Documento recuperado de Dokusi con OID {} versión {}",
					t65bDocumentType.getDocumentID().getId(), t65bDocumentType
							.getDocumentID().getVersion());

			return t65bDocumentType;
		} else {
			DokusiService.LOGGER
					.warn("La respuesta de retrieveDocument de Dokusi es null");
			throw new DokusiException(
					" LOG NEGOCIO: La respuesta de retrieveDocument de Dokusi es null");
		}
	}

	/**
	 * Recupera un documento de Dokusi
	 * 
	 *  auditUser
	 *            String
	 *  doc
	 *            DocumentoOS2
	 *  t65bStringListType
	 *            T65BStringListType
	 * 
	 *  T65BDocumentType.
	 *
	 * @param auditUser the audit user
	 * @param doc the doc
	 * @param t65bStringListType the t 65 b string list type
	 * @return the t 65 B document type
	 * @throws DokusiException             excepción lanzada por Dokusi
	 * @throws AB04BParseException             excepción al parsear la respuesta de Dokusi
	 */
	public T65BDocumentType recuperarDocumento(String auditUser,
			DocumentoOS2 doc, T65BStringListType t65bStringListType)
			throws DokusiException, AB04BParseException {

		DokusiService.LOGGER.info("recuperarDocumento");

		DokusiService.LOGGER.info("auditUser: {}", auditUser);

		DokusiService.LOGGER.info("doc: {}", doc.toString());

		T65BDocumentIDType t65bDocumentIDType = DokusiUtils.getInstance()
				.construirT65BDocumentIDType(doc.getOidDokusi094(), null);
		T65BContentType t65bContentType = DokusiUtils.getInstance()
				.construirT65BContentType(doc);

		String documentID = DokusiUtils.getInstance()
				.getT65BDocumentIDTypeAsString(t65bDocumentIDType);
		String keysList = DokusiUtils.getInstance()
				.getT65BStringListTypeAsString(t65bStringListType);
		String content = DokusiUtils.getInstance().getT65BContentTypeAsString(
				t65bContentType);

		String response = null;
		try {
			DokusiService.LOGGER
					.info("Se va a recuperar de Dokusi el documento con OID {} y versión {}",
							t65bDocumentIDType.getId(),
							t65bDocumentIDType.getVersion());

			response = this.t65bFSDWSClasePortQSPort.retrieveDocument(
					UtilSeguridad.getInstance().getTokenXLNets(), auditUser,
					documentID, keysList, content);
		} catch (Exception e) {
			DokusiService.LOGGER.error(
					"Error en la llamada a retrieveDocument de Dokusi", e);
			throw new DokusiException(e,
					" LOG NEGOCIO: Error en la llamada a retrieveDocument de Dokusi");
		}

		if (response != null) {
			T65BDocumentType t65bDocumentType = DokusiUtils.getInstance()
					.getDokusiResponseAsT65BDocumentType(response);

			DokusiService.LOGGER.info(
					"Documento recuperado de Dokusi con OID {} versión {}",
					t65bDocumentType.getDocumentID().getId(), t65bDocumentType
							.getDocumentID().getVersion());

			return t65bDocumentType;
		} else {
			DokusiService.LOGGER
					.warn("La respuesta de retrieveDocument de Dokusi es null");
			throw new DokusiException(
					" LOG NEGOCIO: La respuesta de retrieveDocument de Dokusi es null");
		}
	}

	/**
	 * Recupera un documento de Dokusi
	 * 
	 *  auditUser
	 *            String
	 *  doc
	 *            DocumentoOS3
	 *  t65bStringListType
	 *            T65BStringListType
	 * 
	 *  T65BDocumentType.
	 *
	 * @param auditUser the audit user
	 * @param doc the doc
	 * @param t65bStringListType the t 65 b string list type
	 * @return the t 65 B document type
	 * @throws DokusiException             excepción lanzada por Dokusi
	 * @throws AB04BParseException             excepción al parsear la respuesta de Dokusi
	 */
	public T65BDocumentType recuperarDocumento(String auditUser,
			DocumentoOS3 doc, T65BStringListType t65bStringListType)
			throws DokusiException, AB04BParseException {

		DokusiService.LOGGER.info("recuperarDocumento");

		DokusiService.LOGGER.info("auditUser: {}", auditUser);

		DokusiService.LOGGER.info("doc: {}", doc.toString());

		T65BDocumentIDType t65bDocumentIDType = DokusiUtils.getInstance()
				.construirT65BDocumentIDType(doc.getOidDokusi097(), null);
		T65BContentType t65bContentType = DokusiUtils.getInstance()
				.construirT65BContentType(doc);

		String documentID = DokusiUtils.getInstance()
				.getT65BDocumentIDTypeAsString(t65bDocumentIDType);
		String keysList = DokusiUtils.getInstance()
				.getT65BStringListTypeAsString(t65bStringListType);
		String content = DokusiUtils.getInstance().getT65BContentTypeAsString(
				t65bContentType);

		String response = null;
		try {
			DokusiService.LOGGER
					.info("Se va a recuperar de Dokusi el documento con OID {} y versión {}",
							t65bDocumentIDType.getId(),
							t65bDocumentIDType.getVersion());

			response = this.t65bFSDWSClasePortQSPort.retrieveDocument(
					UtilSeguridad.getInstance().getTokenXLNets(), auditUser,
					documentID, keysList, content);
		} catch (Exception e) {
			DokusiService.LOGGER.error(
					"Error en la llamada a retrieveDocument de Dokusi", e);
			throw new DokusiException(e,
					" LOG NEGOCIO: Error en la llamada a retrieveDocument de Dokusi");
		}

		if (response != null) {
			T65BDocumentType t65bDocumentType = DokusiUtils.getInstance()
					.getDokusiResponseAsT65BDocumentType(response);

			DokusiService.LOGGER.info(
					"Documento recuperado de Dokusi con OID {} versión {}",
					t65bDocumentType.getDocumentID().getId(), t65bDocumentType
							.getDocumentID().getVersion());

			return t65bDocumentType;
		} else {
			DokusiService.LOGGER
					.warn("La respuesta de retrieveDocument de Dokusi es null");
			throw new DokusiException(
					" LOG NEGOCIO: La respuesta de retrieveDocument de Dokusi es null");
		}
	}

	/**
	 * Transforma un documento de Dokusi a PDFA
	 * 
	 *  auditUser
	 *            String
	 *  t65bDocumentIDType
	 *            T65BDocumentIDType.
	 *
	 * @param auditUser the audit user
	 * @param t65bDocumentIDType the t 65 b document ID type
	 * @throws DokusiException             expcepción lanzada por Dokusi
	 */
	public void transformarPDFA(String auditUser,
			T65BDocumentIDType t65bDocumentIDType) throws DokusiException {

		DokusiService.LOGGER.info("transformarPDFA");

		DokusiService.LOGGER.info("auditUser: {}", auditUser);

		String documentID = DokusiUtils.getInstance()
				.getT65BDocumentIDTypeAsString(t65bDocumentIDType);

		T65BFormatType t65BFormatType = DokusiUtils.getInstance()
				.construirT65BFormatTypePDFA();
		String format = DokusiUtils.getInstance().getT65BFormatTypeAsString(
				t65BFormatType);

		boolean response = false;
		try {
			DokusiService.LOGGER
					.info("Se va a convertir a PDFA en Dokusi al documento con OID {} y versión {}",
							t65bDocumentIDType.getId(),
							t65bDocumentIDType.getVersion());

			response = this.t65bFSDWSClasePortQSPort.createTransformation(
					UtilSeguridad.getInstance().getTokenXLNets(), auditUser,
					documentID, format);

			DokusiService.LOGGER.info(
					"LOG NEGOCIO: transformarPDFA --> devuelve: {}",
					Boolean.toString(response));

			if (!response) {
				throw new DokusiException(
						" LOG NEGOCIO: La llamada a createTransformation ha devuelto error");
			}

		} catch (Exception e) {
			DokusiService.LOGGER.error(
					" LOG NEGOCIO: Error en la llamada a createTransformation",
					e);
			throw new DokusiException(e,
					" LOG NEGOCIO: Error en la llamada a createTransformation");
		}
	}

	/**
	 * Recupera un documento de Dokusi
	 * 
	 *  auditUser
	 *            String
	 *  numRegistro
	 *            String
	 *  procedureId
	 *            String
	 * 
	 *  T65BDocumentIDListType.
	 *
	 * @param auditUser the audit user
	 * @param numRegistro the num registro
	 * @param procedureId the procedure id
	 * @return the t 65 B document ID list type
	 * @throws DokusiException             excepción lanzada por Dokusi
	 * @throws AB04BParseException             excepción al parsear la respuesta de Dokusi
	 */
	public T65BDocumentIDListType recuperarDocumentosSolicitud(
			String auditUser, String numRegistro, String procedureId)
			throws DokusiException, AB04BParseException {

		DokusiService.LOGGER.info("recuperarDocumentosSolicitud - INIT");

		DokusiService.LOGGER.info(
				"recuperarDocumentosSolicitud - auditUser: {}", auditUser);

		DokusiService.LOGGER.info(
				"recuperarDocumentosSolicitud - numRegistro: {}", numRegistro);
		DokusiService.LOGGER.info(
				"recuperarDocumentosSolicitud - procedureId: {}", procedureId);

		T65BConditionListType t65BConditionListType = DokusiUtils.getInstance()
				.generarCondicionesConsulta(numRegistro, procedureId);
		String conditionList = DokusiUtils.getInstance()
				.getT65BConditionListTypeAsString(t65BConditionListType);

		String token = UtilSeguridad.getInstance().getTokenXLNets();

		String response = null;
		try {
			DokusiService.LOGGER
					.info("recuperarDocumentosSolicitud - Se van a recuperar de Dokusi los documentos de la solicitud");

			response = this.t65bFSDWSClasePortQSPort.retrieveDocumentsIdsEx(
					token, auditUser, ConstantesDokusi.TIPO_DOCUMENTAL_ARCHIVO,
					conditionList, true, 0);
		} catch (Exception e) {
			DokusiService.LOGGER
					.error("recuperarDocumentosSolicitud - Error en la llamada a retrieveDocumentsIdsEx de Dokusi",
							e);
			throw new DokusiException(e,
					" LOG NEGOCIO: Error en la llamada a retrieveDocumentsIdsEx de Dokusi");
		}

		if (response != null) {
			T65BDocumentIDListType t65BDocumentIDListType = DokusiUtils
					.getInstance().getDokusiResponseAsT65BDocumentIDListType(
							response);

			DokusiService.LOGGER.info(
					"recuperarDocumentosSolicitud - Recuperados {} documentos",
					t65BDocumentIDListType.getDocumentID().size());

			DokusiService.LOGGER.info("recuperarDocumentosSolicitud", "END");

			return t65BDocumentIDListType;
		} else {
			DokusiService.LOGGER
					.error("recuperarDocumento - La respuesta de retrieveDocumentsIdsEx de Dokusi es null");
			throw new DokusiException(
					" LOG NEGOCIO: La respuesta de retrieveDocumentsIdsEx de Dokusi es null");
		}
	}

}
